
#include "usb_spec.h"
#include "defines.h"
#include "usb_drv.h"
#include "usb_api.h"
#include <avr/io.h>
#include <util/delay.h>
#include <string.h>

/*
  Microcontroller: AT90USB162 TQFP-32
  Register width : 8 bit
  Clock speed    : 16MHz
  GP RAM         : 512 bytes
  USB RAM        : 176 bytes
  Flash          : 16 kbytes
  EEPROM         : 512 bytes

  Pin configuration:

    PB0 -> I/O 0                (in, out)
    PB1 -> I/O 1                (in, out)
    PB2 -> I/O 2                (in, out)
    PB3 -> I/O 3                (in, out)
    PB4 -> I/O 4                (in, out)
    PB5 -> I/O 5                (in, out)
    PB6 -> I/O 6                (in, out)
    PB7 -> I/O 7                (in, out)

    PC2 -> I/O Direction LED #7 (out, active high)
    PC4 -> Activity LED         (out, active high)
    PC5 -> Enabled LED          (out, active high)
    PC6 -> Level Shifter Enable (out, active low, open drain)

    PD0 -> I/O Direction LED #0 (out, active high)
    PD1 -> I/O Direction LED #1 (out, active high)
    PD2 -> I/O Direction LED #2 (out, active high)
    PD3 -> I/O Direction LED #3 (out, active high)
    PD4 -> I/O Direction LED #4 (out, active high)
    PD5 -> I/O Direction LED #5 (out, active high)
    PD6 -> I/O Direction LED #6 (out, active high)
    PD7 -> Program Button       (in,  active low )
*/

// List of configured vendor request messages:
enum {
  USB_VENDOR_REQUEST__CONFIGURE_BIDIRINT__SET_MASKS,
  USB_VENDOR_REQUEST__CONFIGURE_BIDIRINT__SET_STATUS,
  USB_VENDOR_REQUEST__CONFIGURE_BIDIRINT__JTAG_INIT1,
  USB_VENDOR_REQUEST__CONFIGURE_BIDIRINT__JTAG_INIT2,
  USB_VENDOR_REQUEST__CONFIGURE_BIDIRINT__JTAG_DEINIT,

  USB_VENDOR_REQUEST__CONFIGURE_BIDIRINT__BUFFERMODE__WRITESENDBUF,
  USB_VENDOR_REQUEST__CONFIGURE_BIDIRINT__BUFFERMODE__READRECVBUF,
  USB_VENDOR_REQUEST__CONFIGURE_BIDIRINT__BUFFERMODE__SENDDATA,
  USB_VENDOR_REQUEST__CONFIGURE_BIDIRINT__BUFFERMODE__RECVDATA,
  USB_VENDOR_REQUEST__CONFIGURE_BIDIRINT__BUFFERMODE__COMPARESENDRECVBUFS
};

// Global state variables:
unsigned char output_mask, clock_mask, output_no_clock_mask, clock_delay, enabled;
unsigned char jtag_mode, tdi_mask, tms_mask, tck_mask, tdo_mask;

#define TDI_HI() PORTB |=  tdi_mask
#define TDI_LO() PORTB &= ~tdi_mask
#define TMS_HI() PORTB |=  tms_mask
#define TMS_LO() PORTB &= ~tms_mask
#define TCK_HI() PORTB |=  tck_mask
#define TCK_LO() PORTB &= ~tck_mask
#define TDO_RD() (PINB &   tdo_mask)

inline void delay() {
  unsigned char i = clock_delay;
  while( i ) {
    _delay_ms(0.001);
    --i;
  }
}

#include "jtag.c"


inline void SetOutputLEDs(unsigned char mask) {
  PORTD = mask&127;
  if( mask&128 )
    PORTC |=  _BV(PC2);
  else
    PORTC &= ~_BV(PC2);
}
inline void ConfigureIODirections(unsigned char mask) {
  DDRB = mask;
}
inline void ACT_LED_On() {
  PORTC |= _BV(PC4);
}
inline void ACT_LED_Off() {
  PORTC &= ~_BV(PC4);
}
inline void ENA_LED_On() {
  PORTC |= _BV(PC5);
}
inline void ENA_LED_Off() {
  PORTC &= ~_BV(PC5);
}
inline void LevelShifter_Enable() {
  DDRC |=  _BV(PC6);
}
inline void LevelShifter_Disable() {
  DDRC &= ~_BV(PC6);
}


void UsbDevInit() {
  // reset everything to the default state
  DDRB  = 0;
  PORTB = 0;
  DDRC  = _BV(PC2)|_BV(PC4)|_BV(PC5);
  PORTC = 0;
  DDRD  = _BV(PD0)|_BV(PD1)|_BV(PD2)|_BV(PD3)|_BV(PD4)|_BV(PD5)|_BV(PD6);
  PORTD = 0;

  output_mask = 0;
  enabled = 0;
}

void UsbBusyWait() {
  while(1) {
    // do nothing while waiting for USB interrupts
  }
}

unsigned char sendbuf[192], recvbuf[192];
unsigned char sendbufptr, recvbufptr;

void do_send_data() {
  unsigned char* start = sendbuf, * end = sendbuf+192;
  unsigned char clock = clock_mask;
  ACT_LED_On();
  while( start < end ) {
    PORTB = *start;
    PORTB ^= clock;
    asm volatile("nop\nnop");
    ++start;
  }
  PORTB = 0;
  ACT_LED_Off();
}

void do_recv_data() {
  unsigned char* start = recvbuf, * end = recvbuf+192;
  unsigned char clock = clock_mask;
  ACT_LED_On();
  while( start < end ) {
    while( !(PINB&clock) )
      ;
    *start = PINB;
    ++start;
  }
  ACT_LED_Off();
}

void UsbDevProcessVendorRequest(USB_DeviceRequest *req) {
  switch(req->bRequest) {
    case USB_VENDOR_REQUEST__CONFIGURE_BIDIRINT__SET_MASKS:
      output_mask = req->wValue&0xFF;

      SetOutputLEDs(output_mask);

      // make sure no inputs have the pull-up resistor enabled
      PORTB &= output_mask;
      // make sure clocks are always on outputs - no longer
      clock_mask = /*output_mask&(*/req->wValue;//>>8);
      output_no_clock_mask = output_mask & ~clock_mask;

      // configure I/O direction if enabled
      if( enabled )
        ConfigureIODirections(output_mask);
      break;

    case USB_VENDOR_REQUEST__CONFIGURE_BIDIRINT__SET_STATUS:
      clock_delay = req->wValue&0xFF;
      enabled = (req->wValue>>8)&1;

      if( enabled ) {
        // set active LED, turn on level shifter and set I/O directions
        ENA_LED_On();
        LevelShifter_Enable();
        ConfigureIODirections(output_mask);
      } else {
        // turn off active LED, turn off level shifter and turn off outputs
        ENA_LED_Off();
        LevelShifter_Disable();
        ConfigureIODirections(0);
      }
      break;


    case USB_VENDOR_REQUEST__CONFIGURE_BIDIRINT__JTAG_INIT1:
      tdi_mask = req->wValue&0xFF;
      tms_mask = req->wValue>>8;
      break;

    case USB_VENDOR_REQUEST__CONFIGURE_BIDIRINT__JTAG_INIT2:
      tck_mask = req->wValue&0xFF;
      tdo_mask = req->wValue>>8;

      // set I/O directions
      output_mask = tdi_mask|tms_mask|tck_mask;
      ConfigureIODirections(output_mask);
      SetOutputLEDs(output_mask);

      ENA_LED_On();
      LevelShifter_Enable();

      jtag_mode = 1;
      break;

    case USB_VENDOR_REQUEST__CONFIGURE_BIDIRINT__JTAG_DEINIT:
      jtag_mode = 0;

      // turn off outputs
      ConfigureIODirections(0);
      SetOutputLEDs(0);

      // turn off active LED, turn off level shifter and turn off outputs
      ENA_LED_Off();
      LevelShifter_Disable();

      break;

    case USB_VENDOR_REQUEST__CONFIGURE_BIDIRINT__BUFFERMODE__WRITESENDBUF:
      sendbufptr = 1;
      // send data to endpoint 3 now
      break;
    case USB_VENDOR_REQUEST__CONFIGURE_BIDIRINT__BUFFERMODE__READRECVBUF:
      UsbDevSelectEndpoint(4);
      recvbufptr = 0;
      while( recvbufptr != 192 ) {
        unsigned char i;
        while( !UsbDevTransmitterReady() )
          ;
        for( i = 0; i < 64; ++i )
          UsbDevWriteByte(recvbuf[recvbufptr++]);
        UsbDevClearTransmitterReady();
        // Now send it.
        UsbDevSendInData();
      }
      break;
    case USB_VENDOR_REQUEST__CONFIGURE_BIDIRINT__BUFFERMODE__SENDDATA:
      do_send_data();
      break;
    case USB_VENDOR_REQUEST__CONFIGURE_BIDIRINT__BUFFERMODE__RECVDATA:
      do_recv_data();
      break;
    case USB_VENDOR_REQUEST__CONFIGURE_BIDIRINT__BUFFERMODE__COMPARESENDRECVBUFS:
      UsbDevSelectEndpoint(2);
      if( UsbDevTransmitterReady() ) {
        UsbDevWriteByte(memcmp(sendbuf, recvbuf, 192));
        UsbDevClearTransmitterReady();
        // Now send it.
        UsbDevSendInData();
      }
      break;
  }
}

void UsbDevHandleControlMsgOut() {
  // not used
}

void UsbDevHandleControlMsgIn() {
  // not used
}

void UsbDevHandleRecvData() {
  ACT_LED_On();

  if( UsbDevHasReceivedOUT_Data() ) {
    UsbDevClearHasReceivedOUT_Data();
    if( UsbDevReadAllowed() ) {
      unsigned char i;
      unsigned char in_bytes, out_bytes;
      unsigned char in_buf[32], out_buf[32];

      in_bytes = UsbDevGetByteCountLow();

      if( sendbufptr ) {
        for( i = 0; i < in_bytes; ++i ) {
          if( sendbufptr <= 192 ) {
            sendbuf[sendbufptr-1] = UsbDevReadByte();
            ++sendbufptr;
          }
        }
        if( sendbufptr == 193 )
          sendbufptr = 0;
      } if( jtag_mode ) {
        // JTAG mode

        for( i = 0; i < in_bytes; ++i )
          in_buf[i] = UsbDevReadByte();

        out_bytes = do_jtag(in_buf, in_bytes, out_buf);
      } else {
        // bit banging mode

        // Read the data out.
        for( i = 0; i < in_bytes; ++i ) {
          unsigned char byte = UsbDevReadByte();
          PORTB = byte & output_no_clock_mask;
          delay();
          PORTB ^= clock_mask;
          delay();
          out_buf[i] = PINB;
          PORTB ^= clock_mask;
        }

        out_bytes = in_bytes;
      }

      // Tell the chip we're finished with the data we just read.
      UsbDevClearFifoControllBit();

      if( out_bytes ) {
        // Now select Endpoint 4 to send the input bits.
        UsbDevSelectEndpoint(4);
        // We'll send if there's space for it in the buffer, which there generally should be.
        if( UsbDevTransmitterReady() ) {
          for( i = 0; i < out_bytes; ++i )
            UsbDevWriteByte(out_buf[i]);
          UsbDevClearTransmitterReady();
          // Now send it.
          UsbDevSendInData();
        }
      }
    }
  }

  ACT_LED_Off();
}

void UsbDevHandleSendData() {
  //not used
}


void UsbGetStringDescriptor(char s[], uint8_t index) {
  uint8_t i;
#if (USB_MaxStringDescriptorLength < 22)
#error USB_MaxStringDescriptorLength too small!
#endif
  i = USB_MaxStringDescriptorLength;
  while (i--) *s++ = '\0';
  s -= USB_MaxStringDescriptorLength;
  s[1] = USB_StringDescriptorType;
  switch (index)
  {
    case USB_LanguageDescriptorIndex: // == 0
      s[0] = 4;
      s[2] = 9; // two byte language code, only support for English
      s[3] = 4;
      break;
    case USB_ManufacturerStringIndex:
      s[2] = 'S';
      s[4] = 'o';
      s[6] = 'l';
      s[8] = 'i';
      s[10] = 'd';
      s[12] = ' ';
      s[14] = 'C';
      s[16] = 'o';
      s[18] = 'd';
      s[20] = 'e';
      s[0] = 22; // length of descriptor
      break;
    case USB_ProductStringIndex:
      s[2] = 'E';
      s[4] = 'x';
      s[6] = 'a';
      s[8] = 'm';
      s[10] = 'p';
      s[12] = 'l';
      s[14] = 'e';
      s[0] = 16;
      break;
    case USB_SerialNumberStringIndex:
      s[2] = '0';
      s[4] = '1';
      s[6] = '0';
      s[0] = 8;
      break;
    default:
      s[2] = '?';
      s[0] = 4;
  }
}
